#pragma compile(Out, ..\SciTEAbbrevMan.exe)
#pragma compile(Icon, ..\Resources\Main.ico)
#pragma compile(UPX, True)
#pragma compile(Compression, 9)
#pragma compile(Stripper, True)
#pragma compile(FileVersion, 1.8.0.0)
#pragma compile(ProductVersion, 1.8)
#pragma compile(ProductName, SciTEAbbrevMan)
#pragma compile(FileDescription, SciTE Abbreviations Manager)
#pragma compile(LegalCopyright, "Copyright  2011-2019 G.Sandler (CreatoR)")
#pragma compile(CompanyName, CreatoR's Lab)
#pragma compile(Comments, Program made by G.Sandler)

#Au3Stripper_Parameters=/StripOnly /SF=0 SV=0 /CV=0 /CF=0 /CS=0 /CN=0

#Region Header

#CS
	Name:				SciTEAbbrevMan - SciTE Abbrev Manager
	Author:				G.Sandler (CreatoR)
	AutoIt version:		3.3.10.2 - 3.3.14.5
	
	History:
						
						v1.8
						* Program renamed to SciTE Abbrev Manager (SciTEAbbrevMan).
						* Search improved (abbrev content also scanned).
						* Logic of the program has been changed:
							- Now the program works only with au3.abbrev/au3.user.abbrev and au3.keywords.abbreviations.properties/au3.user.keywords.abbreviations.properties files.
							- Now the highlight syntax module take keywords directly from au3.keywords.properties.
							- Now all pathes taken only from config file (wich is renamed to Config.ini), no pathes search performed anymore.
						
						v1.7
						* Program renamed to SCABBRMAN (SciTE Abbreviations Manager).
						* Fixed issue with saving new abbreviation.
						* Now config filename should be the same as executable filename.
						
						v1.6
						* Fixed interface artefacts.
						* Fixed Preview issue.
						* Preview now faster.
						+ Added F1 hotkey for About menu item.
						
						v1.5
						* Program rewritten.
						- Removed SciTE integration (too complicated).
						* Now program settings stored in config ini file (SISCABMAN.ini).
						+ Added ability to set SciTE and abbreviations paths in config file.
						
						v1.4
						* Fixed issue with finding and opening abbreviation files.
						* Internal source code improvements.
						
						v1.3
						+ Added option to search abbreviations in the list. Available also with "Ctrl + F" (search dialog) and with F3 ("Find next" option).
						* Removed skins.
						* Preview option now saved.
						* Few minor fixes.
						
						v1.2
						+ Added "Set on top of all windows" option.
						+ Added "Preview" option. Allows to preview the abbreviation value with highlighted syntax.
						* Fixed an issue with focused skin selection combo.
						* Fixed an issue with save changes confirmation popup message on non selected abbreviation.
						
						v1.1
						+ Added SciTE Integration option.
						+ Added Basic Hotkeys accelerators:
							Del - Delete selected abbreviation.
							Ins - Create new abbreviation.
							CTRL + S - Save modified abbreviation.
						
						* Now the program will remember it's settings (Reg key: HKEY_CURRENT_USER\Software\SISCABMAN).
						* When exit, the program will ask now to save the changes for modified abbreviation.
						* Few minor fixes.
						
						v1.0
						* First release.
	
#CE

#NoTrayIcon
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <StaticConstants.au3>
#include <EditConstants.au3>
#include <ComboConstants.au3>
#include <ListBoxConstants.au3>
#include <GUIListBox.au3>
#include <GUIImageList.au3>
#include <WinAPISys.au3>
#include <File.au3>
#include <IE.au3>

#include "Includes\SciTE.au3"
#include "Includes\Au3SyntaxHighlight.au3"

#EndRegion Header

#Region Global Variables

Global $sApp_Name 					= __GetResData('ProductName')
Global $sApp_Name_Full 				= $sApp_Name & ' - ' & __GetResData('FileDescription')
Global $sApp_Version 				= __GetResData('ProductVersion')
Global $sCopyright					= __GetResData('LegalCopyright')
Global $sConfig_File				= @ScriptDir & (@Compiled ? '' : '\..') & '\Config.ini'
Global $sMainIcon_File				= @Compiled ? @AutoItExe : @ScriptDir & '\' & __GetResData('Icon')
Global $sListIcon_File				= @ScriptDir & (@Compiled ? '' : '\..') & '\Resources\List.ico'

Global $iSetOnTop					= Number(IniRead($sConfig_File, 'Main', 'Set On Top', 0))
Global $iPreview					= Number(IniRead($sConfig_File, 'Main', 'Preview', 0))

Global $sSciTE_Path 				= _PathFull(IniRead($sConfig_File, 'Main', 'SciTE Path', '..\..\'), @ScriptDir)

Global $sAbbrevs_File 				= _PathFull(IniRead($sConfig_File, 'Main', 'Abbrev File', $sSciTE_Path & '\abbrev\au3.abbrev'), @ScriptDir)
Global $sAbbrevsUser_File 			= _PathFull(IniRead($sConfig_File, 'Main', 'Abbrev User File', $sSciTE_Path & '\abbrev\au3.user.abbrev'), @ScriptDir)
Global $sAbbrevsKwrds_File 			= _PathFull(IniRead($sConfig_File, 'Main', 'Abbrev Keywords File', $sSciTE_Path & '\properties\au3.keywords.abbreviations.properties'), @ScriptDir)
Global $sAbbrevsUserKwrds_File 		= _PathFull(IniRead($sConfig_File, 'Main', 'Abbrev User Keywords File', $sSciTE_Path & '\properties\au3.user.keywords.abbreviations.properties'), @ScriptDir)

Global $sAbbrevs_List 				= _SciTE_GetAbbrevList()

Global $fAbbrev_Val_Changed 		= False
Global $sLast_Abbrev_Name			= ''
Global $sLast_Abbrev_Value			= ''
Global $sLast_Search_Abbrev			= ''

$sAu3SH_Styles_File					= @ScriptDir & (@Compiled ? '' : '\..') & '\Resources\au3.styles.properties'

$AU3SH_ADD_URLS						= False
$AU3SH_KEYWORDS_FILE 				= $sSciTE_Path & '\properties\au3.keywords.properties'
$AU3SH_API_FILE 					= $sSciTE_Path & '\api\au3.api'

Global Const $ODT_LISTBOX 			= 2
Global Const $ODA_DRAWENTIRE 		= 0x1
Global Const $ODA_SELECT 			= 0x2
Global Const $ODS_SELECTED 			= 0x1
Global Const $ODS_NOACCEL 			= 0x100
Global Const $ODS_NOFOCUSRECT 		= 0x200

Global Const $iGUI_Width 			= 600
Global Const $iGUI_Height 			= 650

Global $hImageList					= _GUIImageList_Create(16, 16, 5)
_GUIImageList_AddIcon($hImageList, $sListIcon_File)

If WinExists('[CLASS:AutoIt v3;TITLE:~' & $sApp_Name & '~]') Then
	Exit
EndIf

AutoItWinSetTitle('~' & $sApp_Name & '~')

If Not FileExists($sAbbrevs_File) Or Not FileExists($sAbbrevsKwrds_File) Then
	If StringInStr($CmdLineRaw, '/SciTE') Then
		_SciTE_ConsoleWrite('! Abbrevs Or Keywords file not found, please check config.ini.', True, True)
	Else
		MsgBox(16, $sApp_Name, 'Abbrevs Or Keywords file not found, please check config.ini.')
	EndIf
	
	Exit
EndIf

#EndRegion Global Variables

#Region GUI

$hGUI = GUICreate($sApp_Name_Full, $iGUI_Width, $iGUI_Height)
WinSetOnTop($hGUI, '', $iSetOnTop)
GUISetIcon($sMainIcon_File, 0, $hGUI)

$iFile_Menu = GUICtrlCreateMenu('File')
$iOpen_AbbrevFile_MItem = GUICtrlCreateMenuItem('Open Abbrev File', $iFile_Menu)
$iOpen_AbbrevUserFile_MItem = GUICtrlCreateMenuItem('Open Abbrev User File', $iFile_Menu)
GUICtrlCreateMenuItem('', $iFile_Menu)
$iOpen_AbbrevKwrdsFile_MItem = GUICtrlCreateMenuItem('Open Abbrev Keywords File', $iFile_Menu)
$iOpen_AbbrevUserKwrdsFile_MItem = GUICtrlCreateMenuItem('Open Abbrev User Keywords File', $iFile_Menu)
GUICtrlCreateMenuItem('', $iFile_Menu)
$iExit_MItem = GUICtrlCreateMenuItem('Exit', $iFile_Menu)

$iEdit_Menu = GUICtrlCreateMenu('Edit')
$iAddAbbrev_MItem = GUICtrlCreateMenuItem('Add new abbrev...	(INS)', $iEdit_Menu)
$iRemoveAbbrev_MItem = GUICtrlCreateMenuItem('Remove abbrev...	(DEL)', $iEdit_Menu)
$iSaveAbbrev_MItem = GUICtrlCreateMenuItem('Save abbrev	(CTRL + S)', $iEdit_Menu)
$iSearchAbbrev_MItem = GUICtrlCreateMenuItem('Search abbrev...	(F3 / CTRL + F)', $iEdit_Menu)

$iSearchAbbrev_Dummy = GUICtrlCreateDummy()

$iSettings_Menu = GUICtrlCreateMenu('Settings')
$iSetOnTop_MItem = GUICtrlCreateMenuItem('Set on top of all windows', $iSettings_Menu)
If $iSetOnTop Then GUICtrlSetState($iSetOnTop_MItem, $GUI_CHECKED)
$iPreviewAbbrev_MItem = GUICtrlCreateMenuItem('Preview mode', $iSettings_Menu)
If $iPreview Then GUICtrlSetState($iPreviewAbbrev_MItem, $GUI_CHECKED)

$iHelp_Menu = GUICtrlCreateMenu('Help')
$iAbout_MItem = GUICtrlCreateMenuItem('About...', $iHelp_Menu)

GUICtrlCreateLabel('Abbreviations List:', 20, 5, $iGUI_Width - 40, 15, $SS_CENTER)
$iAbbrevs_List = GUICtrlCreateList('', 20, 20, $iGUI_Width - 40, $iGUI_Height - 300, BitOR($WS_BORDER, $WS_VSCROLL, $LBS_HASSTRINGS, $LBS_OWNERDRAWFIXED))
$hAbbrevs_List = GUICtrlGetHandle($iAbbrevs_List)
_GUICtrlListBox_SetData($iAbbrevs_List, $sAbbrevs_List)
GUICtrlSetState($iAbbrevs_List, $GUI_FOCUS)

$iAbbrevValue_Edit = GUICtrlCreateEdit('', 20, $iGUI_Height - 280, $iGUI_Width - 40, 170)

GUICtrlCreateLabel('Hints:', 20, $iGUI_Height - 100, 50, 15)
GUICtrlSetFont(-1, 8, 600)
GUICtrlSetColor(-1, 0xFF0000)
GUICtrlCreateLabel('* Use | symbol to set the cursor position', 50, $iGUI_Height - 80, $iGUI_Width - 70, 15)
GUICtrlSetFont(-1, 8, 600)
GUICtrlCreateLabel('* To insert TAB character, use CTRL + TAB', 50, $iGUI_Height - 65, $iGUI_Width - 70, 15)
GUICtrlSetFont(-1, 8, 600)
GUICtrlCreateLabel('* You can set autoit macros (for ex.: @YEAR@), they will be expanded when abbrev is saved', 50, $iGUI_Height - 50, $iGUI_Width - 70, 15)
GUICtrlSetFont(-1, 8, 600)

$oIE = _IECreateEmbedded()
$iPreviewAbbrev_Obj = _GUICtrlCreateAu3HighlSntxObj($oIE, 20, $iGUI_Height - 280, $iGUI_Width - 40, 160)

GUIRegisterMsg($WM_DRAWITEM, '__WM_DRAWITEM')
GUIRegisterMsg($WM_COMMAND, '__WM_COMMAND')

GUISetState(@SW_SHOW, $hGUI)

#EndRegion GUI

#Region Initializing

_Preview_Proc()

Dim $aAccellKeys[6][2] = _
	[ _
		['^f', $iSearchAbbrev_MItem], ['{F3}', $iSearchAbbrev_Dummy], _
		['{INSERT}', $iAddAbbrev_MItem], ['{DEL}', $iRemoveAbbrev_MItem], _
		['^s', $iSaveAbbrev_MItem], _
		['{F1}', $iAbout_MItem] _
	]

__FixAccelHotKeyLayout()
GUISetAccelerators($aAccellKeys, $hGUI)

#EndRegion Initializing

#Region Main Loop

While 1
	$nMsg = GUIGetMsg()
	
	Switch $nMsg
		Case $GUI_EVENT_CLOSE, $iExit_MItem
			__Exit()
		Case $iOpen_AbbrevFile_MItem
			ShellExecute($sAbbrevs_File)
		Case $iOpen_AbbrevUserFile_MItem
			ShellExecute($sAbbrevsUser_File)
		Case $iOpen_AbbrevKwrdsFile_MItem
			ShellExecute($sAbbrevsKwrds_File)
		Case $iOpen_AbbrevUserKwrdsFile_MItem
			ShellExecute($sAbbrevsUserKwrds_File)
		Case $iSetOnTop_MItem
			;Handled in WM_COMMAND
		Case $iPreviewAbbrev_MItem
			$iPreview = Number(BitAND(GUICtrlRead($iPreviewAbbrev_MItem), $GUI_CHECKED) <> $GUI_CHECKED)
			GUICtrlSetState($iPreviewAbbrev_MItem, $iPreview ? $GUI_CHECKED : $GUI_UNCHECKED)
			_Preview_Proc()
		Case $iAbout_MItem
			MsgBox(64, $sApp_Name & ' - About', $sApp_Name_Full & @CRLF & 'v' & $sApp_Version & @CRLF & @CRLF & $sCopyright, 0, $hGUI)
		Case $iAbbrevs_List
			$sAbbrv_Name = GUICtrlRead($iAbbrevs_List)
			$sAbbrv_Value = _GetAbbrevValue_Proc($sAbbrv_Name)
			
			_SciTE_AbbrevLineToCode($sAbbrv_Value)
			
			If $sAbbrv_Name = $sLast_Abbrev_Name Then
				_PreviewSet_Proc(GUICtrlRead($iAbbrevValue_Edit))
			Else
				_OnAbbrevChanged()
				_PreviewSet_Proc($sAbbrv_Value)
				
				$sLast_Abbrev_Name = $sAbbrv_Name
				$sLast_Abbrev_Value = $sAbbrv_Value
				
				GUICtrlSetData($iAbbrevValue_Edit, $sAbbrv_Value)
			EndIf
		Case $iSearchAbbrev_MItem, $iSearchAbbrev_Dummy
			If $nMsg = $iSearchAbbrev_MItem Or $sLast_Search_Abbrev = '' Then
				$sAbbrv_Name = InputBox('Abbreviation Name', 'Enter abbreviation name or keyword to search inside abbrev value:', $sLast_Search_Abbrev, '', 300, 150, Default, Default, 0, $hGUI)
				If @error Or StringStripWS($sAbbrv_Name, 8) = '' Then ContinueLoop
			Else
				$sAbbrv_Name = $sLast_Search_Abbrev
			EndIf
			
			$sLast_Search_Abbrev = $sAbbrv_Name
			
			$iIndex = _GUICtrlListBox_GetCurSel($iAbbrevs_List)
			$iFind_Abbrev = _GUICtrlListBox_FindInText($iAbbrevs_List, $sAbbrv_Name, $iIndex)
			
			If $iFind_Abbrev = -1 Then
				$aSplit = StringSplit($sAbbrevs_List, '|')
				$iStart = ($iIndex = -1 ? 1 : $iIndex + 2)
				
				For $x = 1 To 2
					For $i = $iStart To $aSplit[0]
						If StringInStr(_GetAbbrevValue_Proc($aSplit[$i]), $sAbbrv_Name) Then
							$iFind_Abbrev = $i - 1
							ExitLoop 2
						EndIf
					Next
					
					$iStart = 1
				Next
				
				If $iFind_Abbrev = -1 Then
					MsgBox(48, 'Attention', 'Abbreviation not found.', 0, $hGUI)
					ContinueLoop
				EndIf
			EndIf
			
			ControlCommand($hGUI, '', $iAbbrevs_List, 'SetCurrentSelection', $iFind_Abbrev)
		Case $iAddAbbrev_MItem
			$sAbbrv_Name = InputBox('Abbreviation Name', 'Enter abbreviation name:', '', '', 300, 150, Default, Default, 0, $hGUI)
			If @error Then ContinueLoop
			
			$iFind_Abbrev = _GUICtrlListBox_FindString($iAbbrevs_List, $sAbbrv_Name, True)
			
			If $iFind_Abbrev <> -1 Then
				MsgBox(48, 'Attention', 'Such abbreviation already exists.', 0, $hGUI)
				ControlCommand($hGUI, '', $iAbbrevs_List, 'SetCurrentSelection', $iFind_Abbrev)
				ContinueLoop
			EndIf
			
			;Write to the keywords
			$sAbbrevs_List &= '|' & $sAbbrv_Name
			_SciTE_PropertiesWrite($sAbbrevsUserKwrds_File, 'au3.keywords.userabbrev', $sAbbrv_Name, 15, 0)
			
			;Write to the user abbrevs file
			$sRead_Abbrev = FileRead($sAbbrevsUser_File)
			
			If Not StringInStr($sRead_Abbrev, '#; -- Templates added by ' & $sApp_Name & ' --') Then
				$sRead_Abbrev &= @CRLF & '#; -- Templates added by ' & $sApp_Name & ' --' & @CRLF
			Else
				$sRead_Abbrev &= @CRLF
			EndIf
			
			$sRead_Abbrev &= $sAbbrv_Name & '='
			
			$hFile = FileOpen($sAbbrevsUser_File, 2)
			FileWrite($hFile, $sRead_Abbrev)
			FileClose($hFile)
			
			$iIndex = _GUICtrlListBox_AddString($iAbbrevs_List, $sAbbrv_Name)
			_GUICtrlListBox_SetItemData($iAbbrevs_List, $iIndex, 0)
			
			ControlCommand($hGUI, '', $iAbbrevs_List, 'SetCurrentSelection', $iIndex)
			GUICtrlSetState($iAbbrevValue_Edit, $GUI_FOCUS)
			
			_SciTE_SendCommand('reloadproperties:')
		Case $iRemoveAbbrev_MItem
			$sAbbrv_Name = GUICtrlRead($iAbbrevs_List)
			
			If $sAbbrv_Name = '' Then
				MsgBox(48, 'Error', 'No abbreviations selected.', 0, $hGUI)
				ContinueLoop
			EndIf
			
			$iAsk = MsgBox(52, 'Attention', 'Are you sure, remove [' & $sAbbrv_Name & ']  abbreviation?', 0, $hGUI)
			
			If $iAsk <> 6 Then
				ContinueLoop
			EndIf
			
			;Delete from the main list
			$sAbbrevs_List = StringRegExpReplace($sAbbrevs_List, '(\|?' & $sAbbrv_Name & '\||\|' & $sAbbrv_Name & '\|?)', '', 1)
			
			;If $sAbbrv_Name found in user abbrevs, delete there, otherwise in standard abbrevs file
			If _IniAction('Read', $sAbbrevsUser_File, 'Abbrevs_Sect', $sAbbrv_Name) Then
				_SciTE_PropertiesWrite($sAbbrevsUserKwrds_File, 'au3.keywords.userabbrev', StringRegExpReplace(_SciTE_PropertiesRead($sAbbrevsUserKwrds_File, 'au3.keywords.userabbrev', '', 0), '(?i)' & $sAbbrv_Name & '(?:\s*|$)', ''), 15, 1)
				
				;Delete from user abbrevs file
				_IniAction('Delete', $sAbbrevsUser_File, 'Abbrevs_Sect', $sAbbrv_Name)
			Else
				_SciTE_PropertiesWrite($sAbbrevsKwrds_File, 'au3.keywords.abbrev', StringRegExpReplace(_SciTE_PropertiesRead($sAbbrevsKwrds_File, 'au3.keywords.abbrev', '', 0), '(?i)' & $sAbbrv_Name & '(?:\s*|$)', ''), 15, 1)
				
				;Delete from abbrevs file
				_IniAction('Delete', $sAbbrevs_File, 'Abbrevs_Sect', $sAbbrv_Name)
			EndIf
			
			_GUICtrlListBox_BeginUpdate($iAbbrevs_List)
			
			$iIndex = _GUICtrlListBox_GetCurSel($iAbbrevs_List)
			_GUICtrlListBox_DeleteString($iAbbrevs_List, $iIndex)
			ControlCommand($hGUI, '', $iAbbrevs_List, 'SetCurrentSelection', _GUICtrlListBox_GetCount($iAbbrevs_List)-1)
			
			_SciTE_SendCommand('reloadproperties:')
			
			_GUICtrlListBox_EndUpdate($iAbbrevs_List)
		Case $iSaveAbbrev_MItem
			$fAbbrev_Val_Changed = False
			
			$sAbbrv_Name = GUICtrlRead($iAbbrevs_List)
			$sAbbrv_Value = GUICtrlRead($iAbbrevValue_Edit)
			
			_SaveAbbrevValue_Proc($sAbbrv_Name, $sAbbrv_Value)
	EndSwitch
WEnd

#EndRegion Main Loop

#Region Program Functions

Func _OnAbbrevChanged($fOnExit = False)
	If Not $fAbbrev_Val_Changed Or $sLast_Abbrev_Value = _GetAbbrevValue_Proc($sLast_Abbrev_Name) Then
		Return 1
	EndIf
	
	$fAbbrev_Val_Changed = False
	
	Local $iAsk = MsgBox($fOnExit ? 51 : 52, 'Attention', 'Abbreviation [' & $sLast_Abbrev_Name & '] has been modified, save the changes?', 0, $hGUI)
	
	Switch $iAsk
		Case 2
			$fAbbrev_Val_Changed = True
			Return 0
		Case 6
			_SaveAbbrevValue_Proc($sLast_Abbrev_Name, $sLast_Abbrev_Value)
	EndSwitch
	
	Return 1
EndFunc

Func _GetAbbrevValue_Proc($sAbbrv_Name)
	Local $sAbbrv_Value = _IniAction('Read', $sAbbrevsUser_File, 'Abbrevs_Sect', $sAbbrv_Name)
	
	If Not $sAbbrv_Value Then
		$sAbbrv_Value = _IniAction('Read', $sAbbrevs_File, 'Abbrevs_Sect', $sAbbrv_Name)
	EndIf
	
	Return $sAbbrv_Value
EndFunc

Func _SaveAbbrevValue_Proc($sAbbrv_Name, $sAbbrv_Value)
	If $sAbbrv_Name = '' Then
		MsgBox(48, 'Error', 'No abbreviation selected.', 0, $hGUI)
		Return
	EndIf
	
	_SciTE_AbbrevCodeToLine($sAbbrv_Value)
	
	_IniAction('Read', $sAbbrevsUser_File, 'Abbrevs_Sect', $sAbbrv_Name)
	
	Local $sFile = (@error ? $sAbbrevs_File : $sAbbrevsUser_File)
	Local $sRead = FileRead($sFile)
	
	_IniAction('Write', $sFile, 'Abbrevs_Sect', $sAbbrv_Name, $sAbbrv_Value)
	_SciTE_SendCommand('reloadproperties:')
EndFunc

Func _Preview_Proc()
	If $iPreview Then
		GUICtrlSetState($iAbbrevValue_Edit, $GUI_HIDE)
		GUICtrlSetState($iPreviewAbbrev_Obj, $GUI_SHOW)
		
		_PreviewSet_Proc(GUICtrlRead($iAbbrevValue_Edit))
	Else
		GUICtrlSetState($iPreviewAbbrev_Obj, $GUI_HIDE)
		GUICtrlSetState($iAbbrevValue_Edit, BitOR($GUI_SHOW, $GUI_FOCUS))
	EndIf
EndFunc

Func _PreviewSet_Proc($sValue)
	If Not $iPreview Or Not $sValue Then
		Return
	EndIf
	
	GUICtrlSetState($iFile_Menu, $GUI_DISABLE)
	GUICtrlSetState($iEdit_Menu, $GUI_DISABLE)
	GUICtrlSetState($iHelp_Menu, $GUI_DISABLE)
	
	_IEDocWriteHTML($oIE, '<font style=''weight:bold;color:#ff0000''>Please wait...</font>')
	
	$AU3SH_HIGHLIGHT_ABORT = False
	
	Local $sAu3Syntax_HighlightedCode = _Au3_SyntaxHighlight($sValue, 0)
	
	If @error Then
		$sAu3Syntax_HighlightedCode = ''
	EndIf
	
	GUICtrlSetState($iFile_Menu, $GUI_ENABLE)
	GUICtrlSetState($iEdit_Menu, $GUI_ENABLE)
	GUICtrlSetState($iHelp_Menu, $GUI_ENABLE)
	
	$AU3SH_HIGHLIGHT_ABORT = False
	
	_IEDocWriteHTML($oIE, $sAu3Syntax_HighlightedCode)
	__IEInsertEventScript($oIE, 'document', 'oncontextmenu', 'return false;')
	GUICtrlSetData($iAbbrevValue_Edit, $sValue)
EndFunc

Func _IniAction($sAction, $sFile, $sSect, $sKey, $sValue = '')
	Local $sRead_File = '[' & $sSect & ']' & @CRLF & FileRead($sFile)
	Local $sRet = '', $iError = 0
	
	Local $hFile = FileOpen($sFile, 2)
	FileWrite($hFile, $sRead_File)
	FileClose($hFile)
	
	Switch $sAction
		Case 'Write'
			IniWrite($sFile, $sSect, $sKey, $sValue)
		Case 'Delete'
			IniDelete($sFile, $sSect, $sKey)
		Case 'Read'
			$sRet = IniRead($sFile, $sSect, $sKey, '')
		Case 'ReadSection'
			$sRet = IniReadSection($sFile, $sSect)
	EndSwitch
	
	$iError = @error
	
	$sRead_File = StringReplace(FileRead($sFile), '[' & $sSect & ']' & @CRLF, '', 1)
	
	$hFile = FileOpen($sFile, 2)
	FileWrite($hFile, $sRead_File)
	FileClose($hFile)
	
	Return SetError($iError, 0, $sRet)
EndFunc

Func _SciTE_GetAbbrevList()
	Local $vAbbrvKwrds_List = _SciTE_PropertiesRead($sAbbrevsKwrds_File, 'au3.keywords.abbrev', '', 0) & @CRLF & _SciTE_PropertiesRead($sAbbrevsUserKwrds_File, 'au3.keywords.userabbrev', '', 0)
	Return StringRegExpReplace($vAbbrvKwrds_List, '\s+', '|')
EndFunc

Func _SciTE_AbbrevCodeToLine(ByRef $sAbbrCode)
	If StringLen($sAbbrCode) = 0 Then Return
	
	$sAbbrCode = __ExpandVarStrings($sAbbrCode)
	$sAbbrCode = StringRegExpReplace($sAbbrCode, '(?<!\\)\\([^\\])', '\\\\$1')
	$sAbbrCode = StringRegExpReplace($sAbbrCode, '\r?\n' , '\\n')
	$sAbbrCode = StringRegExpReplace($sAbbrCode, '(?:\t|\h{3,4})', '\\t')
EndFunc

Func _SciTE_AbbrevLineToCode(ByRef $sAbbrLine)
	If StringLen($sAbbrLine) = 0 Then Return
	
	$sAbbrLine = StringRegExpReplace($sAbbrLine , '(?i)(?<!\\)\\[rn]', @CRLF)
	$sAbbrLine = StringRegExpReplace($sAbbrLine, '(?i)(?<!\\)\\t', @TAB)
EndFunc

Func _SciTE_PropertiesWrite($sFile, $sProperty, $sValue, $iPropsPerLine = 5, $iReplace = 0)
	Local $sFileRead = FileRead($sFile)
	Local $aValue = StringSplit(StringReplace($sValue, '\', ''), '|')
	
	$sValue = ''
	
	For $i = 1 To $aValue[0]
		$sValue &= $aValue[$i]
		
		If $i < $aValue[0] Then
			If ($aValue[0] = 1 Or Mod($i, $iPropsPerLine) = 0) Then
				$sValue &= ' \' & @CRLF & @TAB
			Else
				$sValue &= ' '
			EndIf
		EndIf
	Next
	
	If $iReplace Then
		$sFileRead = StringRegExpReplace($sFileRead, _
			'(?si)(.*?\R?\Q' & $sProperty & '\E)\s*=\s*.*?(\R[^=\R]+=.*|\z)', _
			'\1=' & StringReplace($sValue, '\', '\\') & '\2', 1)
	Else
		$sFileRead = StringRegExpReplace($sFileRead, _
			'(?si)(.*?\R?\Q' & $sProperty & '\E)\s*=\s*(.*)', _
			'\1=' & StringReplace($sValue, '\', '\\') & ' \\' & @CRLF & @TAB & '\2', 1)
	EndIf
	
	If @extended = 0 Then
		$sFileRead &= @CRLF & $sProperty & '=' & $sValue
	EndIf
	
	$hFile = FileOpen($sFile, 2)
	FileWrite($hFile, $sFileRead)
	Return FileClose($hFile)
EndFunc

Func _SciTE_PropertiesRead($sFile, $sProperty, $sDefault = '', $iRetArr = 1, $iDelDups = 0)
	Local $aFileRead = StringSplit(StringStripCR(FileRead($sFile)), @LF)
	Local $sRet, $iPropLen = StringLen($sProperty)
	
	For $i = 1 To $aFileRead[0]
		$aFileRead[$i] = StringStripWS($aFileRead[$i], 3)
		
		If StringLeft($aFileRead[$i], $iPropLen + 1) = $sProperty & '=' Then
			$aFileRead[$i] = StringTrimLeft($aFileRead[$i], $iPropLen + 1)
			
			If StringRight($aFileRead[$i], 1) = '\' Then
				$aFileRead[$i] = StringTrimRight($aFileRead[$i], 1)
			EndIf
			
			$sRet &= StringStripWS($aFileRead[$i], 3)
			
			For $j = $i + 1 To $aFileRead[0]
				If StringInStr($aFileRead[$j], '=') Then
					ExitLoop 2
				ElseIf StringLeft($aFileRead[$j], 1) <> '#' Then
					If StringRight($aFileRead[$j], 1) = '\' Then
						$aFileRead[$j] = StringTrimRight($aFileRead[$j], 1)
					EndIf
					
					$sRet &= ' ' & StringStripWS($aFileRead[$j], 3)
				EndIf
			Next
			
			ExitLoop
		EndIf
	Next
	
	If $sRet = '' Then
		$sRet = $sDefault
	EndIf
	
	$sRet = StringStripWS($sRet, 3)
	
	If $iRetArr Then
		$sRet = StringSplit($sRet, ' ' & @TAB)
		
		If $iDelDups Then
			_ArrayDeleteDuplicates($sRet)
		EndIf
	EndIf
	
	Return $sRet
EndFunc

Func _ArrayDeleteDuplicates(ByRef $aArray)
	Local $oDict, $aRet[UBound($aArray)]
	
	$oDict = ObjCreate('Scripting.Dictionary')
	
	If Not IsObj($oDict) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$oDict.CompareMode = 1 ;Text mode (not case sensitive)
	
	For $i = 1 To UBound($aArray)-1
		If Not $oDict.Exists($aArray[$i]) Then
			$oDict.Add($aArray[$i], $aArray[$i])
			
			$aRet[0] += 1
			$aRet[$aRet[0]] = $aArray[$i]
		EndIf
	Next
	
	ReDim $aRet[$aRet[0] + 1]
	$aArray = $aRet
	
	Return 1
EndFunc

Func _GUICtrlCreateAu3HighlSntxObj($oIE, $iLeft, $iTop, $iWidth, $iHeight)
	Local $iPreviewAbbrev_Obj = GUICtrlCreateObj($oIE, $iLeft, $iTop, $iWidth, $iHeight)
	GUICtrlSetState($iPreviewAbbrev_Obj, $GUI_HIDE)
	_IENavigate($oIE, 'about:blank')
	Return $iPreviewAbbrev_Obj
EndFunc

Func _GUICtrlListBox_SetData($iCtrlID, $sData)
	GUICtrlSetData($iCtrlID, $sData)
	
	For $i = 0 To _GUICtrlListBox_GetCount($iCtrlID) - 1
		_GUICtrlListBox_SetItemData($iCtrlID, $i, 0)
	Next
EndFunc

Func __ExpandVarStrings($sString, $iMode = -1)
	Local $aExtractStrs, $sPattern, $sEval, $Old_Opt_EVS
	Local $sRetStr = $sString, $sExpndStr = ''
	Local $aSymbExpnd[2] = ['$', '@']
	
	Local $Old_Func_Opt_EVS = Opt('ExpandVarStrings', 0)
	
	For $i = 0 To 1
		Switch $iMode
			Case 1
				$sPattern = '(?i)(\' & $aSymbExpnd[$i] & '[0-9a-z_]+)'
			Case 2
				$sPattern = '(?i)(\' & $aSymbExpnd[$i] & '[0-9a-z_]+\' & $aSymbExpnd[$i] & ')|(\' & $aSymbExpnd[$i] & '[0-9a-z_]+)'
			Case Else
				$sPattern = '(?i)(\' & $aSymbExpnd[$i] & '[0-9a-z_]+\' & $aSymbExpnd[$i] & ')'
		EndSwitch
		
		Local $aExtractStrs = StringRegExp($sString, $sPattern, 3)
		If Not IsArray($aExtractStrs) Then ContinueLoop
		
		For $iE = 0 To UBound($aExtractStrs) - 1
			If $i = 0 Then
				$sEval = Eval(StringReplace($aExtractStrs[$iE], $aSymbExpnd[$i], '', 0, 2))
				
				If $sEval <> '' Then
					$sRetStr = StringReplace($sRetStr, $aExtractStrs[$iE], $sEval, 0, 2)
				EndIf
			ElseIf $iMode <> -1 Or StringRight($aExtractStrs[$iE], 1) = $aSymbExpnd[$i] Then
				$sExpndStr = $aExtractStrs[$iE]
				
				If StringRight($sExpndStr, 1) <> $aSymbExpnd[$i] Then
					$sExpndStr &= $aSymbExpnd[$i]
				EndIf
				
				$Old_Opt_EVS = Opt('ExpandVarStrings', 1)
				$sExpndStr = $sExpndStr
				Opt('ExpandVarStrings', $Old_Opt_EVS)
				
				If $aExtractStrs[$iE] <> '' Then
					$sRetStr = StringReplace($sRetStr, $aExtractStrs[$iE], $sExpndStr, 0, 2)
				EndIf
			EndIf
		Next
	Next
	
	Opt('ExpandVarStrings', $Old_Func_Opt_EVS)
	Return $sRetStr
EndFunc

Func __IEInsertEventScript($oObject, $sHtmlFor, $sEvent, $sScript)
	If Not IsObj($oObject) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $oHead = $oObject.document.all.tags('HEAD').Item(0)
	Local $oScript = $oObject.document.createElement('script')
	
	If Not IsObj($oScript) Or Not IsObj($oHead) Then
		Return SetError(1, 0, 0)
	EndIf
	
	With $oScript
		.defer = True
		.language = 'jscript'
		.type = 'text/javascript'
		.htmlFor = $sHtmlFor
		.event = $sEvent
		.text = $sScript
	EndWith
	
	$oHead.appendChild($oScript)
	Return SetError(0, 0, 1)
EndFunc

Func __FixAccelHotKeyLayout()
	Static $iKbrdLayout, $aKbrdLayouts
	
	If Execute('@exitMethod') <> '' Then
		Local $iUnLoad = 1
		
		For $i = 1 To $aKbrdLayouts[0]
			If Hex($iKbrdLayout) = Hex('0x' & StringRight($aKbrdLayouts[$i], 4)) Then
				$iUnLoad = 0
				ExitLoop
			EndIf
		Next
		
		If $iUnLoad Then
			_WinAPI_UnloadKeyboardLayout($iKbrdLayout)
		EndIf
		
		Return
	EndIf
	
	$iKbrdLayout = 0x0409
	$aKbrdLayouts = _WinAPI_GetKeyboardLayoutList()
	_WinAPI_LoadKeyboardLayout($iKbrdLayout, $KLF_ACTIVATE)
	
	OnAutoItExitRegister('__FixAccelHotKeyLayout')
EndFunc

Func __GetResData($sRes)
	If @Compiled Then
		Return FileGetVersion(@AutoItExe, $sRes)
	EndIf
	
	Local $sRet = StringRegExpReplace(FileRead(@ScriptFullPath), '(?si)^.*?#pragma compile\(' & $sRes & ', [''"]?(.*?)[''"]?\)\r\n.*$', '\1')
	
	If @extended = 0 Then
		$sRet = StringRegExpReplace(FileRead(@ScriptFullPath), '(?si)^.*?#AutoIt3Wrapper_Res_' & $sRes & '=([^\r\n]+).*$', '\1')
		
		If @extended = 0 Then
			Return $sRes
		EndIf
	EndIf
	
	Return StringStripWS($sRet, 3)
EndFunc

Func __WM_COMMAND($hWnd, $nMsg, $wParam, $lParam)
	Local $nNotifyCode = BitShift($wParam, 16)
	Local $nID = BitAND($wParam, 0xFFFF)
	
	Switch $nID
		Case $iAbbrevValue_Edit
			Switch $nNotifyCode
				Case $EN_CHANGE, $EN_UPDATE
					$fAbbrev_Val_Changed = True
					$sLast_Abbrev_Value = GUICtrlRead($iAbbrevValue_Edit)
			EndSwitch
		Case $iAbbrevs_List
			Switch $nNotifyCode
				Case $LBN_SELCHANGE
					If $iPreview Then
						$AU3SH_HIGHLIGHT_ABORT = True
					EndIf
			EndSwitch
		Case $iSetOnTop_MItem
			$iSetOnTop = Number(BitAND(GUICtrlRead($iSetOnTop_MItem), $GUI_CHECKED) <> $GUI_CHECKED)
			GUICtrlSetState($iSetOnTop_MItem, ($iSetOnTop ? $GUI_CHECKED : $GUI_UNCHECKED))
			WinSetOnTop($hGUI, '', $iSetOnTop)
		Case $iPreviewAbbrev_MItem
			If $iPreview And BitAND(GUICtrlRead($iPreviewAbbrev_MItem), $GUI_CHECKED) = $GUI_CHECKED Then
				$AU3SH_HIGHLIGHT_ABORT = True
			EndIf
	EndSwitch
	
	Return $GUI_RUNDEFMSG
EndFunc

Func __WM_DRAWITEM($hWnd, $Msg, $wParam, $lParam)
	Local $tagDRAWITEMSTRUCT, $cID, $itmID, $itmAction, $itmState, $hItm, $hDC, $iBrushColor, $hBrush, $hBrushOld
	$tagDRAWITEMSTRUCT = DllStructCreate("uint cType;uint cID;uint itmID;uint itmAction;uint itmState;hwnd hItm;hwnd hDC;int itmRect[4];dword itmData", $lParam)
	
	If DllStructGetData($tagDRAWITEMSTRUCT, "cType") <> $ODT_LISTBOX Then
		Return $GUI_RUNDEFMSG
	EndIf
	
	$cID = DllStructGetData($tagDRAWITEMSTRUCT, "cID")
	$itmID = DllStructGetData($tagDRAWITEMSTRUCT, "itmID")
	$itmAction = DllStructGetData($tagDRAWITEMSTRUCT, "itmAction")
	$itmState = DllStructGetData($tagDRAWITEMSTRUCT, "itmState")
	$hItm = DllStructGetData($tagDRAWITEMSTRUCT, "hItm")
	$hDC = DllStructGetData($tagDRAWITEMSTRUCT, "hDC")
	
	Switch $itmAction
		Case $ODA_DRAWENTIRE, $ODA_SELECT
			If $itmState = $ODS_SELECTED Or $itmState = BitOR($ODS_SELECTED, $ODS_NOACCEL, $ODS_NOFOCUSRECT) Then
				$iBrushColor = _WinAPI_GetSysColor($COLOR_HIGHLIGHT)
			Else
				$iBrushColor = 0xFFFFFF ;_WinAPI_GetSysColor($COLOR_HIGHLIGHTTEXT)
			EndIf
			
			$hBrush = _WinAPI_CreateSolidBrush($iBrushColor)
			$hBrushOld = _WinAPI_SelectObject($hDC, $hBrush)

			_WinAPI_FillRect($hDC, DllStructGetPtr($tagDRAWITEMSTRUCT, "itmRect"), $hBrush)
			_WinAPI_SelectObject($hDC, $hBrushOld)
			_WinAPI_DeleteObject($hBrush)
			_WinAPI_SetBkMode($hDC, $TRANSPARENT)
			
			Local $itmText = _GUICtrlListBox_GetText($hItm, $itmID)
			Local $itmTextIMG = _GUICtrlListBox_GetItemData($hItm, $itmID)
			
			Local $tRECT = DllStructCreate("int Left;int Top;int Right;int Bottom")
			DllStructSetData($tRECT, "Left", DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 1) + 20)
			DllStructSetData($tRECT, "Top", DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 2) + 1)
			DllStructSetData($tRECT, "Right", DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 3))
			DllStructSetData($tRECT, "Bottom", DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 4))
			
			_WinAPI_DrawText($hDC, $itmText, $tRECT, $DT_LEFT)
			
			Local $iX = DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 1) + 1
			Local $iY = DllStructGetData($tagDRAWITEMSTRUCT, "itmRect", 2) + 1
			Local $fIsSelected = ($itmState = $ODS_SELECTED) * 2
			
			If $itmTextIMG >= 0 Then
				_GUIImageList_Draw($hImageList, $itmTextIMG, $hDC, $iX, $iY, BitOR(1, $fIsSelected))
			EndIf
	EndSwitch
	
	Return $GUI_RUNDEFMSG
EndFunc

Func __Exit()
	If Not _OnAbbrevChanged(True) Then
		Return
	EndIf
	
	IniWrite($sConfig_File, 'Main', 'Set On Top', $iSetOnTop)
	IniWrite($sConfig_File, 'Main', 'Preview', $iPreview)
	
	GUIDelete($hGUI)
	
	Exit
EndFunc

#EndRegion Program Functions
